/*****************************************************************************
 * prodos/dentry.c
 * dentry operations.
 *
 * Apple II ProDOS Filesystem Driver for Linux 2.4.x
 * Copyright (c) 2001 Matt Jensen.
 * This program is free software distributed under the terms of the GPL.
 *
 * 25-May-2001: Created
 *****************************************************************************/

/*= Kernel Includes =========================================================*/

#include <linux/time.h>
#include <linux/fs.h>

/*= ProDOS Includes =========================================================*/

#include "prodos.h"

/*= Forward Declarations ====================================================*/

/* For prodos_dentry_operations. */
int prodos_hash_dentry(struct dentry *,struct qstr *);
int prodos_compare_dentry(struct dentry *,struct qstr *,struct qstr *);

/*= Preprocessor Macros =====================================================*/

/*= VFS Interface Structures ================================================*/

struct dentry_operations prodos_dentry_operations = {
	d_hash: prodos_hash_dentry,
	d_compare: prodos_compare_dentry
};

/*= Private Functions =======================================================*/

/*= Interface Functions =====================================================*/

/*****************************************************************************
 * prodos_hash_dentry()
 * Check a name for validity under ProDOS naming rules; if it is valid, hash
 * it.
 *****************************************************************************/
int prodos_hash_dentry(struct dentry *de,struct qstr *qstr) {
	unsigned long hash = 0;
	int len = 0;
	int i = 0;

	/* Ensure that we have a valid ProDOS name. */
	i = prodos_check_name(qstr);
	if (i) return i;

	/* Hash the name. */
	len = PRODOS_MIN(qstr->len,PRODOS_MAX_NAME_LEN);
	hash = init_name_hash();
	for (i = 0;i < len;i++)
		hash = partial_name_hash(PRODOS_TOUPPER(qstr->name[i]),hash);
	qstr->hash = end_name_hash(hash);
	return 0;
}

/*****************************************************************************
 * prodos_compare_dentry()
 * Compare two hashed ProDOS names.
 *****************************************************************************/
int prodos_compare_dentry(struct dentry *de,struct qstr *q1,struct qstr *q2) {
	int len = 0;
	int i = 0;

	/* q2 has not been validated yet. */
	i = prodos_check_name(q2);
	if (i) return i;

	/* Ignore characters past the maximum ProDOS name length. */
	len = q1->len;
	if (q1->len >= PRODOS_MAX_NAME_LEN) {
		if (q2->len < PRODOS_MAX_NAME_LEN) return 1;
		len = PRODOS_MAX_NAME_LEN;
	}
	else if (len != q2->len)
		return ~0;

	/* Compare the strings. */
	for (i = 0;i < len;i++) {
		if (PRODOS_TOUPPER(q1->name[i]) != PRODOS_TOUPPER(q2->name[i]))
			return ~0;
	}
	return 0;
}

